// Copyright (c) 2000
//      Maurice Castro (maurice@serc.rmit.edu.au)
//      All rights reserved.
// 
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions
// are met:
// 
// 1. Redistributions of source code must retain the above copyright
//    notice, this list of conditions and the following disclaimer.
// 2. Redistributions in binary form must reproduce the above copyright
//    notice, this list of conditions and the following disclaimer in the
//    documentation and/or other materials provided with the distribution.
// 
// This software is provided by the original author and contributors ``as
// is'' and any express or implied warranties, including, but not limited
// to, the implied warranties of merchantability and fitness for a
// particular purpose are disclaimed.  In no event shall the authors or
// contributors be liable for any direct, indirect, incidental, special,
// exemplary, or consequential damages (including, but not limited to,
// procurement of substitute goods or services; loss of use, data, or
// profits; or business interruption) however caused and on any theory of
// liability, whether in contract, strict liability, or tort (including
// negligence or otherwise) arising in any way out of the use of this
// software, even if advised of the possibility of such damage.
//
// $Id: ikskey.cpp,v 1.13 2000/03/09 04:29:51 maurice Exp $

#include "ikskey.h"

//
// EXPORTed functions
//

// This function is the entry point for the application code.
// It creates an instance of the CApaApplication-derived class,
// CSkeyApplication.
EXPORT_C CApaApplication* NewApplication()
{
	return new CSkeyApplication;
}

// This function is required by all EPOC32 DLLs. In this example,
// it does nothing.
GLDEF_C TInt E32Dll(TDllReason)
{
	return KErrNone;
}


////////////////////////////////////////////////////////////////
//
// Application class, CSkeyApplication
//
////////////////////////////////////////////////////////////////

// This function is used by the EIKON framework to inquire the
// application's UID. KUidIkskey is defined in ikskey.h, and its
// value matches that in the project definition file, ikskey.mmp.
TUid CSkeyApplication::AppDllUid() const
{
	return KUidIkskey;
}

// This function is called by the EIKON framework at application
// start-up. It creates an instance of the document class.
CApaDocument* CSkeyApplication::CreateDocumentL()
{
	return new (ELeave) CSkeyDocument(*this);
}


////////////////////////////////////////////////////////////////
//
// Document class, CSkeyDocument
//
////////////////////////////////////////////////////////////////

CSkeyDocument::CSkeyDocument(CEikApplication& aApp): CEikDocument(aApp)
{
}

// This function is called by the EIKON framework as soon as the 
// document has been created. It creates an instance of the App UI
// class.
CEikAppUi* CSkeyDocument::CreateAppUiL()
{
	return new(ELeave) CSkeyAppUi;
}

CSkeyDocument::~CSkeyDocument()
{
	// release all of the STDLIB resources associated with this thread
	// Satisfies the CONE policy that the heap should be balanced across 
	// the lifetime of a CCoeEnv.
	CloseSTDLIB();
}

////////////////////////////////////////////////////////////////
//
// App UI class, CSkeyAppUi
//
////////////////////////////////////////////////////////////////

// Two-phase construction of the App UI is required, because 
// the App UI creates the application view, which is a C-class.
void CSkeyAppUi::ConstructL()
{
	BaseConstructL();
	iAppView=new(ELeave) CSkeyAppView;
	AddToStackL(iAppView);
	iAppView->ConstructL(ClientRect());
}

// C++ destructor - this releases the resources allocated in ConstructL()
CSkeyAppUi::~CSkeyAppUi()
{
	TRAPD(error, CloseHelpL());
	delete iAppView;
}

// This function is called by the EIKON framework whenever the user
// invokes a command. EEikCmdExit is a standard EIKON command, 
// defined in EIKCMDS.HRH.
void CSkeyAppUi::HandleCommandL(TInt aCommand)
{
	switch (aCommand)
	{
		case EEikCmdExit: 
			Exit();
			break;
		case EEikCmdHelpContents:
			OpenHelpL();
			break;
		case ESkeyAbout:
			CreateDialogL();
			break;
		case ESkeySKEY:
			iAppView->iMode = SKEY;
			iAppView->Update();
			break;
		case ESkeyOTPMD4:
			iAppView->iMode = OTPMD4;
			iAppView->Update();
			break;
		case ESkeyOTPMD5: 
			iAppView->iMode = OTPMD5;
			iAppView->Update();
			break;
		case ESkeyOTPSHA1:
			iAppView->iMode = OTPSHA1;
			iAppView->Update();
			break;
		case ESkeySECRET:
			if (iAppView->iPass->iMode == SecretEditor::Secret)
				iAppView->iPass->iMode = SecretEditor::Obscure;
			else
				iAppView->iPass->iMode = SecretEditor::Secret;
			iAppView->iPass->Reset();
			iAppView->Update();
			break;
	}
}

void CSkeyAppUi::CreateDialogL()
{
	CSkeyDialog* dialog;
	// Create the dialog
	dialog = new (ELeave) CSkeyDialog();
	// Launch the dialog
	dialog->ExecuteLD(R_SKEY_ABOUT_DIALOG);
}


void CSkeyAppUi::DynInitMenuPaneL(TInt aMenuId,CEikMenuPane* aMenuPane)
{
	if (aMenuId==R_SKEY_OPTION_MENU)
	{
		switch (iAppView->iMode)
		{
			case SKEY:
				aMenuPane->SetItemButtonState(ESkeySKEY, EEikMenuItemSymbolOn);
				break;
			case OTPMD4:
				aMenuPane->SetItemButtonState(ESkeyOTPMD4, EEikMenuItemSymbolOn);
				break;
			case OTPMD5:
				aMenuPane->SetItemButtonState(ESkeyOTPMD5, EEikMenuItemSymbolOn);
				break;
			case OTPSHA1:
				aMenuPane->SetItemButtonState(ESkeyOTPSHA1, EEikMenuItemSymbolOn);
				break;
		}
		switch (iAppView->iPass->iMode)
		{
			case SecretEditor::Secret:
				aMenuPane->SetItemButtonState(ESkeySECRET, EEikMenuItemSymbolOn);
				break;
			case SecretEditor::Obscure:
				break;
		}
	}
}

// See C++ Knowledge Base - 
// How can I bring an existing instance of help to the foreground?

void CSkeyAppUi::OpenHelpL()
{
	// find the Apps's name and directory
	TFileName appName(Application()->AppFullName());
	TParse parser;
	User::LeaveIfError(parser.SetNoWild(_L(".hlp"),&appName, NULL));
	TPtrC helpFile (parser.FullName());
	if (! ConeUtils::FileExists(helpFile)) 
		User::Leave(KErrNotFound);

	// now check to see if it is open
	TPtrC helpFileName (parser.Name());
	HBufC *caption = HBufC::NewLC(helpFileName.Length()+2);
	caption->Des().Format(_L("- %S"), &helpFileName);
	TApaTaskList taskList (iEikonEnv->WsSession());
	TInt numGroups = iEikonEnv->WsSession().NumWindowGroups(0);

	// look through applist to see what we can find
	for (TInt index=0; index < numGroups; index++) 
	{
		TApaTask *task= new (ELeave) TApaTask(taskList.FindByPos(index));
		CleanupStack::PushL(task);
		CApaWindowGroupName* wgName=CApaWindowGroupName::NewLC(iEikonEnv->WsSession(),task->WgId());
		TUid uid = wgName->AppUid();
		TPtrC taskCaption = wgName->Caption();
		TBool found = uid.iUid == 0x10000171 && taskCaption.FindF(*caption)>=0; // help app with the desired caption
		if (found) 
		{
			task->BringToForeground();
			CleanupStack::PopAndDestroy(3); // wgName, task and caption
			return; // exit procedure
		}
		CleanupStack::PopAndDestroy(2); // wgName and task
	}
	CleanupStack::PopAndDestroy(); // caption

	// otherwise open using EikDll
	EikDll::StartDocL(helpFile);
	User::After(1000000); // pause 1s
}

void CSkeyAppUi::CloseHelpL()
{
	// find the Apps's name and directory
	TFileName appName(Application()->AppFullName());
	TParse parser;
	User::LeaveIfError(parser.SetNoWild(_L(".hlp"),&appName, NULL));
	TPtrC helpFile (parser.FullName());
	if (! ConeUtils::FileExists(helpFile)) 
		User::Leave(KErrNotFound);

	// now check to see if it is open
	TPtrC helpFileName (parser.Name());
	HBufC *caption = HBufC::NewLC(helpFileName.Length()+2);
	caption->Des().Format(_L("- %S"), &helpFileName);
	TApaTaskList taskList (iEikonEnv->WsSession());
	TInt numGroups = iEikonEnv->WsSession().NumWindowGroups(0);

	// look through applist to see what we can find
	for (TInt index=0; index < numGroups; index++) 
	{
		TApaTask *task= new (ELeave) TApaTask(taskList.FindByPos(index));
		CleanupStack::PushL(task);
		CApaWindowGroupName* wgName=CApaWindowGroupName::NewLC(iEikonEnv->WsSession(),task->WgId());
		TUid uid = wgName->AppUid();
		TPtrC taskCaption = wgName->Caption();
		TBool found = uid.iUid == 0x10000171 && taskCaption.FindF(*caption)>=0; // help app with the desired caption
		if (found) 
		{
			task->EndTask();
			CleanupStack::PopAndDestroy(3); // wgName, task and caption
			return; // exit procedure
		}
		CleanupStack::PopAndDestroy(2); // wgName and task
	}
	CleanupStack::PopAndDestroy(); // caption
}

////////////////////////////////////////////////////////////////
//
// Application view class, CSkeyAppView
//
////////////////////////////////////////////////////////////////

// Two-phase construction is required for the application view,
// because it has to allocate all the resources required by
// Draw().
void CSkeyAppView::ConstructL(const TRect& aRect)
{
	// Make this a window-owning control
	CreateWindowL();
	// initialise
	iMode = SKEY;
	// set up label env
	_LIT(KArial, "Arial");
        TFontSpec spec1(KArial, 330);
        iTitleFont = iCoeEnv->CreateScreenFontL(spec1);
        TFontSpec spec2(KArial, 270);
        iHeadFont = iCoeEnv->CreateScreenFontL(spec2);
	// set up labels
	iText1 = iCoeEnv->AllocReadResourceL(R_SKEY_TEXT);
	iTitle = CreateHeadingL(iText1);
	iTitle->SetFont(iTitleFont);
	iText2 = iCoeEnv->AllocReadResourceL(R_CHLG_TEXT);
	iChTitle = CreateHeadingL(iText2);
	iChTitle->SetFont(iHeadFont);
	iText3 = iCoeEnv->AllocReadResourceL(R_RESP_TEXT);
	iRespTitle = CreateHeadingL(iText3);
	iRespTitle->SetFont(iHeadFont);
	iText4 = iCoeEnv->AllocReadResourceL(R_CNT_TEXT);
	iCntLabel = CreateHeadingL(iText4);
	iText5 = iCoeEnv->AllocReadResourceL(R_SEQ_TEXT);
	iSeqLabel = CreateHeadingL(iText5);
	iText6 = iCoeEnv->AllocReadResourceL(R_PASS_TEXT);
	iPassLabel = CreateHeadingL(iText6);
	// Create the active controls
	iPass = new SecretEditor();
	iPass->SetContainerWindowL(*this);
	iPass->SetMaxLength(45);
	iPass->SetObserver(this);
	iPass->iMode = SecretEditor::Secret;
	iCnt = CreateCntL(0,9999);
	iFocused = iCnt;
	iFocused->SetFocus(ETrue,EDrawNow);
	iSeq = CreateEdL(20,255,(CEikEdwin::TFlags) 0);
	iResp = CreateEdL(20,32,CEikEdwin::EReadOnly);
	iRespNum = CreateEdL(12,20,CEikEdwin::EReadOnly);
	// Set its size
	SetRectL(aRect);
	ActivateL();
}

// The application view's destructor should release all resources
// allocated in ConstructL().
CSkeyAppView::~CSkeyAppView()
{
	iCoeEnv->ReleaseScreenFont(iTitleFont);
	iCoeEnv->ReleaseScreenFont(iHeadFont);
	delete iText1;
	delete iTitle;
	delete iText2;
	delete iChTitle;
	delete iText3;
	delete iRespTitle;
	delete iText4;
	delete iCntLabel;
	delete iText5;
	delete iSeqLabel;
	delete iText6;
	delete iPassLabel;
	delete iPass;
	delete iCnt;
	delete iResp;
	delete iRespNum;
	delete iSeq;
}

// generate a heading
CEikLabel* CSkeyAppView::CreateHeadingL(TDesC *text)
{
	CEikLabel* label = new(ELeave) CEikLabel;
	CleanupStack::PushL(label);
	label->SetContainerWindowL(*this);
	label->SetTextL(*text);
	label->SetAlignment(EHCenterVCenter);
	label->SetSizeL(label->MinimumSize());
	CleanupStack::Pop();
	label->SetNonFocusing();
	return label;
}

// generate a number editor
CEikNumberEditor* CSkeyAppView::CreateCntL(TInt aMin, TInt aMax)
{
	CEikNumberEditor* editor = new(ELeave) CEikNumberEditor;
	CleanupStack::PushL(editor);
	editor->SetContainerWindowL(*this);
	editor->ConstructL(aMin,aMax,aMin);
	editor->SetSizeL(editor->MinimumSize());
	editor->SetObserver(this);
	CleanupStack::Pop();
	return editor;
}

// generate a editor window
CEikEdwin* CSkeyAppView::CreateEdL(TInt aWidth, TInt aText, CEikEdwin::TFlags aFlg)
{
	CEikEdwin* editor = new(ELeave) CEikEdwin;
	CleanupStack::PushL(editor);
	editor->SetContainerWindowL(*this);
	editor->ConstructL(aFlg,aWidth,aText,1);
	editor->SetSizeL(editor->MinimumSize());
	editor->SetObserver(this);
	CleanupStack::Pop();
	return editor;
}

TInt CSkeyAppView::CountComponentControls() const
{
	return 11;
}

CCoeControl* CSkeyAppView::ComponentControl(TInt aIndex) const
{
	switch (aIndex)
	{
		case 0: return(iTitle); 
		case 1: return(iChTitle); 
		case 2: return(iRespTitle); 
		case 3: return(iCntLabel);
		case 4: return(iSeqLabel);
		case 5: return(iPassLabel);
		case 6: return(iPass);
		case 7: return(iCnt);
		case 8: return(iSeq);
		case 9: return(iResp);
		case 10: return(iRespNum);
	}
	return 0;
}

void CSkeyAppView::SizeChangedL()
{
	// We have to layout each component here
	// get size of control
	TInt w=Size().iWidth;
	TInt h=Size().iHeight;
	TInt sep;
	// compute locations starting from here at the center
	TInt x= w / 2;
	TInt y= h / 8;
	TPoint pos(x,y);
	// position title
	PlaceCentL(iTitle, pos);
	// position title 2
	PlaceCentL(iChTitle, pos);
	pos.iX = w / 8;
	sep = w / 6;
	PlaceSBSL(iCntLabel, iCnt, pos, sep);
	PlaceSBSL(iSeqLabel, iSeq, pos, sep);
	PlaceSBSL(iPassLabel, iPass, pos, sep);
	pos.iX = x;
	// position title 3
	PlaceCentL(iRespTitle, pos);
	PlaceCentL(iResp, pos);
	PlaceCentL(iRespNum, pos);
}

void CSkeyAppView::PlaceCentL(CCoeControl *ctrl, TPoint &pos)
{
	TSize size = ctrl->MinimumSize();
	pos.iX += -(size.iWidth/2);
	ctrl->SetExtentL(pos, size);
	pos.iX += (size.iWidth/2);
	pos.iY += (size.iHeight);
}

void CSkeyAppView::PlaceSBSL(CCoeControl *ctrl1, CCoeControl *ctrl2, TPoint &pos, TInt sep)
{
	TPoint pos2 = pos;
	pos2.iX += sep;
	TSize size = ctrl1->MinimumSize();
	ctrl1->SetExtentL(pos, size);
	TSize size2 = ctrl2->MinimumSize();
	ctrl2->SetExtentL(pos2, size2);
	if ((size.iHeight) < (size2.iHeight))
		pos.iY += size2.iHeight;
	else
		pos.iY += size.iHeight;
}

// This function draws the application view on the screen.
// Note that it MUST NOT leave.
void CSkeyAppView::Draw(const TRect& /*aRect*/) const
{
}

TKeyResponse CSkeyAppView::OfferKeyEventL(const TKeyEvent& aKeyEvent,TEventCode aType)
{
	// treat Enter key specially ... it causes both recalculation and 
	// movement between fields
	// treat arrow keys specially they move between fields
	if (aType==EEventKey)
	{
		switch (aKeyEvent.iCode)
		{
			case EKeyEnter:
				// dont rotate to response field from pass
				if (iFocused != iPass)
				{
					Rotate(+1);
				}
				// if focus on pass copy password and obscure
				if (iFocused == iPass)
				{
					iPass->OfferKeyEventL(aKeyEvent, aType);
				}
				Update();
				return(EKeyWasConsumed);
			case EKeyUpArrow:
				Rotate(-1);
				return(EKeyWasConsumed);
			case EKeyDownArrow:
				Rotate(+1);
				return(EKeyWasConsumed);
		}
	}
	// distribute keypresses to component with focus
	if (iFocused)
	{
		return iFocused->OfferKeyEventL(aKeyEvent, aType);
	}
	return EKeyWasNotConsumed;
}

void CSkeyAppView::Rotate(TInt aDir)
{
	if (aDir > 0)
	{
		if (iFocused == iCnt)
			ChFocus(iSeq);
		else if (iFocused == iSeq)
			ChFocus(iPass);
		else if (iFocused == iPass)
			ChFocus(iResp);
		else if (iFocused == iResp)
			ChFocus(iRespNum);
	}
	else
	{
		if (iFocused == iRespNum)
			ChFocus(iResp);
		if (iFocused == iResp)
			ChFocus(iPass);
		else if (iFocused == iPass)
			ChFocus(iSeq);
		else if (iFocused == iSeq)
			ChFocus(iCnt);
	}
}

void CSkeyAppView::ChFocus(CCoeControl* aControl)
{
	iFocused->SetFocus(EFalse,EDrawNow);
	iFocused=aControl;
	iFocused->SetFocus(ETrue,EDrawNow);
}

// track focus among components
void CSkeyAppView::HandleControlEventL(CCoeControl* aControl, TCoeEvent aEventType)
{
	switch (aEventType)
	{
		case EEventRequestFocus:
			if (iFocused!=aControl)
			{
				ChFocus(aControl);
			}
			break;
		default:
			break;
	}

}

void CSkeyAppView::CompSKey()
{
	TBufC<33> res;
	TBufC<22> res2;
        char buf[34];
        char buf2[21];
	TBuf<256> password;
	TBuf<256> seq;
	iSeq->GetText(seq);
	iPass->GetText(password);
	// call the C routine provided by the engine
	enc(iCnt->Number(), seq.PtrZ(), password.PtrZ(), buf, buf2, iMode);
	res = (const unsigned char *) buf;
	res2 = (const unsigned char *) buf2;
	iResp->SetTextL(&res);
	iResp->DrawNow();
	iRespNum->SetTextL(&res2);
	iRespNum->DrawNow();
}

void CSkeyAppView::Update()
{
	CompSKey();
}

////////////////////////////////////////////////////////////////
//
// Dialog implemenation, CSkeyDialog
//
////////////////////////////////////////////////////////////////

CSkeyDialog::CSkeyDialog()
{
}

void CSkeyDialog::PreLayoutDynInitL()
{
}

TBool CSkeyDialog::OkToExitL(TInt /*aKeycode*/)
{
	return ETrue;
}
