// Copyright (c) 2000
//      Maurice Castro (maurice@serc.rmit.edu.au)
//      All rights reserved.
// 
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions
// are met:
// 
// 1. Redistributions of source code must retain the above copyright
//    notice, this list of conditions and the following disclaimer.
// 2. Redistributions in binary form must reproduce the above copyright
//    notice, this list of conditions and the following disclaimer in the
//    documentation and/or other materials provided with the distribution.
// 
// This software is provided by the original author and contributors ``as
// is'' and any express or implied warranties, including, but not limited
// to, the implied warranties of merchantability and fitness for a
// particular purpose are disclaimed.  In no event shall the authors or
// contributors be liable for any direct, indirect, incidental, special,
// exemplary, or consequential damages (including, but not limited to,
// procurement of substitute goods or services; loss of use, data, or
// profits; or business interruption) however caused and on any theory of
// liability, whether in contract, strict liability, or tort (including
// negligence or otherwise) arising in any way out of the use of this
// software, even if advised of the possibility of such damage.
//
// $Id: seced.cpp,v 1.2 2000/12/14 02:53:43 maurice Exp $

/* This code is based on the EIKON SecretEditor Code */
/* it has been modified to allow for more characters in the buffer, */
/* to allow the message to be typed in the clear and subsequently obscured */
/* by pressing the enter key, or to implement the same behaviour as the */
/* original secure editor */

/* Names and comments have been made as similar as possible to ensure that */
/* the code can be compared easily */

#include <barsread.h>
#include <baclipb.h>
#include <eikenv.h>
#include <e32keys.h>
#include <txtetext.h>
#include "seced.h"

#include <eikcolor.h>

EXPORT_C SecretEditor::SecretEditor(): CEikBorderedControl(TEikBorder(TEikBorder::ESingleGray))
{
	iSecCharArr.SetLength(iSecCharArr.MaxLength());
	iBuf.SetLength(iBuf.MaxLength());
	iFont=iCoeEnv->NormalFont();
	iCharWidth=iFont->WidthZeroInPixels();
	iAscent=iFont->AscentInPixels();
	iCr = EFalse;
	iMode = Secret;
	iSecretChar = '*';
}

// returns the minimum size needed to display
EXPORT_C TSize SecretEditor::MinimumSize()
{
	TSize size=iBorder.SizeDelta();
	size.iHeight+=iCoeEnv->NormalFont()->HeightInPixels();
	// 1 pixel margin
	size.iHeight+=2;
	// room for cursor at the end + 1 pixel gap on each side
	size.iWidth+=iCharWidth*(iMaxLen+1)+2; 
	return size;
}

// Respond to key presses
EXPORT_C TKeyResponse SecretEditor::OfferKeyEventL(const TKeyEvent& aKeyEvent,TEventCode /*aType*/)
{
	TInt code=aKeyEvent.iCode;
	if ((code==EKeyUpArrow) || (code==EKeyDownArrow))
		return(EKeyWasNotConsumed);
	if (code==EKeyEnter)
                iCr = ETrue;
	if (TChar(code).IsPrint())
	{
		if (iCr)
		{
			Reset();
		}
		if (iSecPos<iMaxLen)
		{
			iSecCharArr[iSecPos]=(TText)code;
			iSecPos++;
		}
	}
	else if (code==EKeyBackspace && iSecPos>0)
	{
		iSecPos--;
		iSecCharArr[iSecPos]=0;
		if (iCr)
		{
			Reset();
		}
	}
	else if (code==HotKeyPaste)
	{
		// note: should really check modifier state as well 
		PasteL();
	}
	if ((iMode == Secret) || (iCr))
		iBuf.Fill(iSecretChar,iSecPos);
	else
		iBuf = iSecCharArr;
        DrawTextNow();
        DisplayCursor();
	ReportEventL(MCoeControlObserver::EEventStateChanged);
	return EKeyWasConsumed;
}

// Get the actual text typed
EXPORT_C void SecretEditor::GetText(TDes& aText) const
{
	aText=iSecCharArr.Left(iSecPos);
}

// Clear the text
EXPORT_C void SecretEditor::Reset()
{
	iSecPos=0;
	iSecCharArr.FillZ(); // obliterate any trace of password in memory
	iBuf.FillZ();
	iCr = EFalse;
	DrawTextNow();
	DisplayCursor();
}

// Set the maximum number of characters for the secret editor
EXPORT_C void SecretEditor::SetMaxLength(TInt aMaxLength)
{
	iMaxLen=aMaxLength;
	__ASSERT_ALWAYS(iMaxLen<=iSecCharArr.MaxLength(), User::Panic(_L("SecEd"), 0) );
}

// Draw or remove cursor as required depending on whether we have the focus
EXPORT_C void SecretEditor::FocusChanged(TDrawNow /*aDrawNow*/)
{
	if (IsFocused())
		DisplayCursor();
	else 
		iEikonEnv->HideCursor(this);
}

// Draw the whole secret editor
EXPORT_C void SecretEditor::Draw(const TRect& /*aRect*/) const
{
	iBorder.Draw(SystemGc(), Rect());
	DrawText();
}

// Draw everything inside the border flicker free
void SecretEditor::DrawText() const
{
	CWindowGc& gc=SystemGc();
	TRect rect=iBorder.InnerRect(Rect());
	rect.iTl.iX++; //leave one pixel gap between the border and text
	gc.SetBrushStyle(CGraphicsContext::ESolidBrush);
	// KEikSecretEditorBackgroundColor
	gc.SetBrushColor(iEikonEnv->ControlColor(EEikColorWindowBackground,*this)); 
	gc.SetPenColor(iEikonEnv->ControlColor(EEikColorWindowText,*this));
	gc.UseFont(iFont);
	TSize size=iBorder.SizeDelta();
	gc.DrawText(iBuf,rect,iAscent+size.iHeight,CGraphicsContext::ELeft,0);
}

void SecretEditor::DrawTextNow() const
{
	ActivateGc();
	DrawText();
	DeactivateGc();
}

// Draw the cursor at the appropriate position
void SecretEditor::DisplayCursor()
{
	if (!IsFocused())
		return;
	TPoint pos=Position();
	TMargins margins=iBorder.Margins();
	pos.iX+=margins.iLeft+iFont->TextWidthInPixels(iBuf)+1;
	//subtract one for one pixel gap on all sides ??? what !?!
	pos.iY+=iAscent+2;	
	iEikonEnv->DrawCursor(this,pos,iCharWidth);
}

void SecretEditor::PasteL()
{
	CClipboard *cb = NULL;

	// Create a clipboard.
	TRAPD(ret,cb=CClipboard::NewForReadingL(iCoeEnv->FsSession()));
	CleanupStack::PushL(cb);
	if (ret==KErrPathNotFound || ret==KErrNotFound)
	{
		User::Leave(ret);
	}

	// Check if there is anything to retrieve off the clipboard.
	// We hope that applications which generate rich text will always
	// put a plain text on the clipboard as well. Word seems to.
	TStreamId stid = (cb->StreamDictionary()).At(KClipboardUidTypePlainText);
	if (stid == KNullStreamId)
	{
		User::Leave(0);
	}

	// Make an object to hold the text
	CPlainText *item = CPlainText::NewL();
	CleanupStack::PushL(item);

	// Insert clipboard into Text
	item->PasteFromStoreL(cb->Store(),cb->StreamDictionary(),0);

	// Copy item to character array
	// Note that this loop is not fully tested as I haven't
	// run into any segmented buffers small enough to test with
	int start = 0;
	while(start < item->DocumentLength())
	{
		int length = MaxSecEdLength - iSecPos;
		TPtrC v = item->Read(start, length);
		for (int j = 0; j < v.Length(); j++)
		{
			iSecCharArr[iSecPos]=v[j];
			start++;
			iSecPos++;
			if (start >= item->DocumentLength())
				break;
		}
	}

	// Delete Clipboard and Text
	CleanupStack::PopAndDestroy(2);
}
