/*
 * Copyright (c) 2000
 *      Maurice Castro (maurice@serc.rmit.edu.au)
 *      All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 
 * This software is provided by the original author and contributors ``as
 * is'' and any express or implied warranties, including, but not limited
 * to, the implied warranties of merchantability and fitness for a
 * particular purpose are disclaimed.  In no event shall the authors or
 * contributors be liable for any direct, indirect, incidental, special,
 * exemplary, or consequential damages (including, but not limited to,
 * procurement of substitute goods or services; loss of use, data, or
 * profits; or business interruption) however caused and on any theory of
 * liability, whether in contract, strict liability, or tort (including
 * negligence or otherwise) arising in any way out of the use of this
 * software, even if advised of the possibility of such damage.
 */

/* this code implements FIPS 180-1. It produces a digest consisiting of 5 */
/* 32 bit ints in system dependent endianness */

/* parts of this code have been based on the sample md4/md5 implementations */

/* $Id: sha1c.c,v 1.1 2000/03/07 00:59:25 maurice Exp $ */

#include <assert.h>
#include "global.h"
#include "sha1.h"

static UINT4 f PROTO_LIST((int, UINT4, UINT4, UINT4));
static UINT4 k PROTO_LIST((int));
static UINT4 s PROTO_LIST((int, UINT4));
static void SHA1Transform PROTO_LIST((UINT4 h[5], unsigned char block[64]));
static void SHA1_memcpy PROTO_LIST((POINTER output, POINTER input, unsigned int len));
static void SHA1_memset PROTO_LIST((POINTER output, int value, unsigned int len));
static void Decode PROTO_LIST((UINT4 *output, unsigned char *input, unsigned int len));
static void Encode PROTO_LIST((unsigned char *output, UINT4 *input, unsigned int len));

static const unsigned char PADDING[64] = {
  0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0
};

void SHA1Init (context)
SHA1_CTX *context;
{
	context->count[0] = context->count[1] = 0;
	/* Load magic initialization constants. */
	context->h[0] = 0x67452301;
	context->h[1] = 0xEFCDAB89;
	context->h[2] = 0x98BADCFE;
	context->h[3] = 0x10325476;
	context->h[4] = 0xC3D2E1F0;
	context->digest[0] = 0x0;
	context->digest[1] = 0x0;
	context->digest[2] = 0x0;
	context->digest[3] = 0x0;
	context->digest[4] = 0x0;
}

void SHA1Update (context, input, inputLen)
SHA1_CTX *context;                                       /* context */
unsigned char *input;                                /* input block */
unsigned int inputLen;                     /* length of input block */
{
	unsigned int i, index, partLen;

	/* Compute number of bytes mod 64 */
	index = (unsigned int)((context->count[0] >> 3) & 0x3F);

	/* Update number of bits */
	if ((context->count[0] += ((UINT4)inputLen << 3)) < ((UINT4)inputLen << 3))
		context->count[1]++;
	context->count[1] += ((UINT4)inputLen >> 29);

	partLen = 64 - index;

	/* Transform as many times as possible. */
	if (inputLen >= partLen) 
	{
		SHA1_memcpy ((POINTER)&context->buffer[index], (POINTER)input, partLen);
		SHA1Transform (context->h, context->buffer);

		for (i = partLen; i + 63 < inputLen; i += 64)
			SHA1Transform (context->h, &input[i]);

		index = 0;
	}
	else
		i = 0;

	/* Buffer remaining input */
	SHA1_memcpy ((POINTER)&context->buffer[index], (POINTER)&input[i], inputLen-i);
}

/* SHA1 finalization. Ends an SHA1 message-digest operation, writing the
  the message digest and zeroizing the context.  */
void SHA1Final (context)
SHA1_CTX *context;                                       /* context */
{
	unsigned char bits[8];
	unsigned int index, padLen;

	/* Save number of bits */
	Encode (&bits[0], &context->count[1], 4);
	Encode (&bits[4], &context->count[0], 4);

	/* Pad out to 56 mod 64. */
	index = (unsigned int)((context->count[0] >> 3) & 0x3f);
	padLen = (index < 56) ? (56 - index) : (120 - index);
	SHA1Update (context, PADDING, padLen);

	/* Append length (before padding) */
	SHA1Update (context, bits, 8);
	/* Store state in digest */
	SHA1_memcpy((POINTER)context->digest, (POINTER)context->h, 20);

	/* Zeroize sensitive information. */
	SHA1_memset ((POINTER)context->h, 1, sizeof (context->h));
	SHA1_memset ((POINTER)context->count, 1, sizeof (context->count));
	SHA1_memset ((POINTER)context->buffer, 1, sizeof (context->buffer));
}

/* Note: Replace "for loop" with standard memcpy if possible. */
static void SHA1_memcpy (output, input, len)
POINTER output;
POINTER input;
unsigned int len;
{
	unsigned int i;

	for (i = 0; i < len; i++)
		output[i] = input[i];
}

static UINT4 s(n, x) 
int n;
UINT4 x;
{
	return(x << n) | (x >> (32-(n)));
}

static UINT4 f(t, b, c, d)
int t;
UINT4 b, c, d;
{
	assert(0 <= t && t <= 79);
	if ( 0 <= t && t <= 19) 
		return((b & c) | ((~ b) & d));
	if (20 <= t && t <= 39) 
		return(b ^ c ^ d);
	if (40 <= t && t <= 59) 
		return((b & c) | (b & d) | (c & d));
	if (60 <= t && t <= 79)
		return(b ^ c ^ d);
	/* this should not occur */
	return(0);
}

static UINT4 k(t)
int t;
{
	assert(0 <= t && t <= 79);
	if ( 0 <= t && t <= 19) 
		return(0x5A827999);
	if (20 <= t && t <= 39) 
		return(0x6ED9EBA1);
	if (40 <= t && t <= 59) 
		return(0x8F1BBCDC);
	if (60 <= t && t <= 79) 
		return(0xCA62C1D6);
	/* this should not occur */
	return(0);
}

/* SHA1 basic transformation. Transforms state based on block. */
static void SHA1Transform (h, block)
UINT4 h[5];
unsigned char block[64];
{
	UINT4 a, b, c, d, e;
	UINT4 w[80];
	UINT4 temp;
	int t;

	Decode (w, block, 64);

	for (t = 16; t <= 79; t++)
	{
	 	w[t] = s(1, w[t-3] ^ w[t-8] ^ w[t-14] ^ w[t-16]); 
	}

	a = h[0]; b = h[1]; c = h[2]; d = h[3]; e = h[4];

	for (t = 0; t <= 79; t++)
	{
                  temp = s(5, a) + f(t, b,c,d) + e + w[t] + k(t); 
                  e = d; d = c; c = s(30, b); b = a; a = temp; 
	}

	h[0] = h[0] + a; h[1] = h[1] + b; h[2] = h[2] + c; 
	h[3] = h[3] + d; h[4] = h[4] + e;
}

/* Note: Replace "for loop" with standard memset if possible. */
static void SHA1_memset (output, value, len)
POINTER output;
int value;
unsigned int len;
{
	unsigned int i;

	for (i = 0; i < len; i++)
		((char *)output)[i] = (char)value;
}

/* Decodes input (unsigned char) into output (UINT4). Assumes len is
  a multiple of 4.
 */
static void Decode (output, input, len)
UINT4 *output;
unsigned char *input;
unsigned int len;
{
	unsigned int i, j;

	for (i = 0, j = 0; j < len; i++, j += 4)
		output[i] = ((UINT4)input[j+3]) | (((UINT4)input[j+2]) << 8) |
		(((UINT4)input[j+1]) << 16) | (((UINT4)input[j]) << 24);
}

/* Encodes input (UINT4) into output (unsigned char). Assumes len is
   a multiple of 4.  */
static void Encode (output, input, len)
unsigned char *output;
UINT4 *input;
unsigned int len;
{
  unsigned int i, j;

  for (i = 0, j = 0; j < len; i++, j += 4) {
    output[j+3] = (unsigned char)(input[i] & 0xff);
    output[j+2] = (unsigned char)((input[i] >> 8) & 0xff);
    output[j+1] = (unsigned char)((input[i] >> 16) & 0xff);
    output[j] = (unsigned char)((input[i] >> 24) & 0xff);
  }
}
